/* Adam Ryan, Michael Henderson 2007
 * This program is a project for the REU Site 2007 research program. 
 * This code was originally adapted from the book "Multicast Sockets: Practical Guide for Programmers" 
 * by David Makofske and Kevin Almeroth
 */

using System;             //For console and exception classes
using System.Net;         //For IPAddress Class
using System.Net.Sockets; //For Socket Class
using System.IO;          //For IO Class

/// <summary>
/// Mutlicast Receiver Class
/// Version 1.0.0 Beta v3
/// Allows for receiving multicast packets of (text) and sending of the pseudo NACKS using Multicasting
/// </summary>
class MulticastReceiver
{
    const int Min_Port = 1024;      //the minimum port value
    const int Max_Port = 65534;     //the maximum port value

    public static void Main(string[] args)
    {
        Boolean    done    = false; //loop variable
        Boolean    turnOff = false; //If this is on then the seq and pCount wont be always equal
        int        Max_Len = 1024;  //maximum receive buffer size
        int        mcastPort;       //destination port
        int        seq;             //Error checking sequence
        int        pCount = 0;          //This is what the sequence is being compared to
        IPAddress  mcastIP;         //destination multicast address
        IPAddress  sourceIP;        //the unicast address of the source
        IPEndPoint receivePt;       //IP endpoint
        Socket     mcastSock;       //multicast socket
        String     now;             //Gives the current time
        TextWriter tw;              //For recording results
        MulticastReceiver mr = new MulticastReceiver(); //To enable the use of methods

        //verify the correct usage of command line arguments
        if (args.Length != 3)
        {
            Console.Error.WriteLine("Usage: MCReceive " + "<Multicast IP> <Multicast Port> <Unicast IP of Source>");
            return;
        }

        //validate the input multicast IP address
        try
        {
            mcastIP = IPAddress.Parse(args[0]);
        }
        catch (Exception)
        {
            Console.Error.WriteLine("That IP Address is not valid");
            return;
        }
            
        //Validate the port number
        try
        {
            mcastPort = Int32.Parse(args[1]);
        }
        catch (Exception)
        {
            Console.Error.WriteLine("That is not a valid port.");
            return;
        }
        if ((mcastPort < Min_Port) || (mcastPort > Max_Port))
        {
            Console.Error.WriteLine("That port is not in the correct range.");
            Console.Error.WriteLine("The port must be between " + Min_Port + " and " + Max_Port);
            return;
        }
        
        //Validate the input for the unicast IP address of the source 
        try
        {
            sourceIP = IPAddress.Parse(args[2]);
        }
        //ummm...not sure about what we're writing in this next bit...
        catch (Exception e)
        {
            Console.Error.WriteLine(e.ToString() + " Is not a valid IP Address for your source.");
            return;
        }

        try
        {
            //Initialize TextWriter
            tw = new StreamWriter("C:\\mc_receive_log.txt", true);

            //Sets the "now" string
            now = DateTime.Now.Hour + ":" + DateTime.Now.Minute + ":" + DateTime.Now.Second + ":" + DateTime.Now.Millisecond;

            //create the socket
            mcastSock = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);

            //set the reuse address option
            mcastSock.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, 1);

            //creates an IPEndPoint and bind to it--perhaps it needs to be IPAddress.Any
            IPEndPoint iPEP = new IPEndPoint(IPAddress.Any, mcastPort);
            mcastSock.Bind(iPEP);

            //Add membership in the multicast group--this was modified from the original 
            //to accept data only from a single source instead of from any source.
            //hopefully we can get it to actually work correctly to only do from one source
            mcastSock.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.AddMembership,
                                    new MulticastOption(mcastIP));

            //creates the endpoint class--this is for receiving
            //first parameter was IPAddress.Any--trying sourceIP first, then mcastIP
            //if we do sourceIP, it'll create a unicast; the problem with using the mcastIP
            //is that single source issues?
            receivePt = new IPEndPoint(mcastIP, mcastPort);    //changed from first parameter of sourceIP
            EndPoint tempReceivePt = (EndPoint)receivePt;

            tw.WriteLine(DateTime.Now + "\r\n");
            tw.Flush();

            //notifies the user that the program has connected to the socket and is waiting for data
            Console.WriteLine("Listening to {0} on port {1} from the source {2}", mcastIP, mcastPort, sourceIP);
            Console.WriteLine("Waiting to receive data.\n");

            //loop to actually receive the data
            while (!done)
            {
                byte[] receivedData = new byte[Max_Len];

                //receive the multicast packets  
                int length = mcastSock.ReceiveFrom(receivedData, 0, Max_Len, SocketFlags.None,
                                                ref tempReceivePt);
                //Updates the "now" string
                now = DateTime.Now.Hour + ":" + DateTime.Now.Minute + ":" + DateTime.Now.Second + ":" + DateTime.Now.Millisecond;

                //format and output the received data packet
                System.Text.ASCIIEncoding encode = new System.Text.ASCIIEncoding();

                //Sets the seq and pCount to when the receiver joins the multicast group
                seq = mr.getSequence(encode.GetString(receivedData, 0, length));

                pCount = mr.initCount(seq, pCount, turnOff); //Just to make sure pCount and seq increment seperately

                //Reforms the received data without the sequence
                string output = mr.stripSeq(encode.GetString(receivedData, 0, length));

                if (seq != pCount)
                {
                    //acknowledgement stuff--to respond to successfully receiving data
                    string a = "NACK";
                    byte[] ack = encode.GetBytes(a);
                    IPEndPoint ackPt = new IPEndPoint(sourceIP, 65535);
                    mcastSock.SendTo(ack, 0, ack.Length, SocketFlags.None, ackPt);

                    //for debugging--shows the ack being encoding/decoded correctly
                    Console.WriteLine("<Sent {0}>", encode.GetString(ack));
                    tw.WriteLine("<Sent {0}>", encode.GetString(ack));
                    tw.Flush();
                }
                else
                {
                    Console.WriteLine("<Received " + length + " bytes from " + tempReceivePt.ToString() + " at " + now + ">: " +
                                        output + "\r\n");
                    tw.WriteLine("<Received " + length + " bytes from " + tempReceivePt.ToString() + " at " + now + ">: " +
                                        output + "\r\n");
                    tw.Flush();
                }

                //Update the counter
                pCount++;

                //Raises the turnOff flag for the initCount method
                turnOff = true;
            }

            //drops membership from ALL sources.  this could be modified to make it drop
            //only a single source if need be--replace IPAddress.Any with IPAddress.sourceIP
            //taking out the second parameter altogether right now
            mcastSock.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.DropMembership,
                                    new MulticastOption(mcastIP));

            //closes the socket
            mcastSock.Close();
        }

        //some basic error handling
        catch (SocketException se)
        {
            Console.Error.WriteLine("Socket Exception: " + se.ToString());
            return;
        }
        catch (IOException ioe)
        {
            Console.Error.WriteLine("IO Exception: " + ioe.ToString());
            return;
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("Exception: " + e.ToString());
            return;
        } 
        return;
    }

    //Retrieves the sequence for the ACK message
    private int getSequence(string data)
    {
        try
        {
            char[] splitter = { ':' };
            string[] tokens = data.Split(splitter);

            return Int32.Parse(tokens[(tokens.Length - 1)]);
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("Exception: " + e.ToString());
            return 0;
        }
    }

    //Takes the sequence off and returns the original string
    private string stripSeq(string data)
    {
        try
        {
            string message = "";

            char[] splitter = { ':' };
            string[] tokens = data.Split(splitter);

            for (int i = 0; i < tokens.Length - 1; i++)
            {
                message += tokens[i];
            }

            return message;
        }
        catch (Exception e)
        {
            Console.WriteLine("Exception: " + e.ToString());
            return null;
        }
    }

    //Initializes pCount (this is meant to be a one time run through, subsequent run throughs will just return the counter)
    private int initCount(int num, int dummy, bool off)
    {
        try
        {
            if (!off)
            {
                return num;
            }
            else
                return dummy;
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("Exception: " + e.ToString());
            return 0;
        }
    }
}